package cachesim;
import java.io.*;
import cachesim.MemoryHandler;
import cachesim.ReferenceData;

class OutputData {
	private int refNum;
	private int instrNum;
	private int dataNum;
	private int readNum;
	private int writeNum;
	private int instrMissNum;
	private int dataMissNum;
	private int readMissNum;
	private int writeMissNum;
	
	public OutputData (int rfn, int in, int dn, int rn, int wn, int imn, int dmn, int rmn, int wmn) {
		refNum = rfn;
		instrNum = in;
		dataNum = dn;
		readNum = rn;
		writeNum = wn;
		instrMissNum = imn;
		dataMissNum = dmn;
		readMissNum = rmn;
		writeMissNum = wmn;
	}
	
	public int getRefNum () {
		return refNum;
	}
	
	public int getInstrNum () {
		return instrNum;
	}
	
	public int getDataNum () {
		return dataNum;
	}
	
	public int getReadNum () {
		return readNum;
	}
	
	public int getWriteNum () {
		return writeNum;
	}
	
	public int getInstrMissNum () {
		return instrMissNum;
	}
	
	public int getDataMissNum () {
		return dataMissNum;
	}
	
	public int getReadMissNum () {
		return readMissNum;
	}
	
	public int getWriteMissNum () {
		return writeMissNum;
	}
}

abstract class MappingElement {
}

class AssociativeMappingElement extends MappingElement {
	private int tag;
	private int address;
	
	public AssociativeMappingElement (int t, int a) {
		tag = t;
		address = a;
	}
	
	public AssociativeMappingElement (int elem, int blockdim, int out) {
		tag = elem / blockdim;
		address = elem % blockdim;
	} 
	
	public int getTag () {
		return tag;
	}
	
	public int getAddress () {
		return address;
	}
	
	public void setTag (int t) {
		tag = t;
	}
	
	public void setAddress (int a) {
		address = a;
	}
}

class DirectMappingElement extends MappingElement {
	private int tag;
	private int block;
	private int address;
	
	public DirectMappingElement (int t, int b, int a) {
		tag = t;
		block = b;
		address = a;
	}
	
	public DirectMappingElement (int elem, int cachedim, int blockdim, int out) {
		tag = elem / cachedim;
		block = (elem % cachedim) / blockdim;
		address = elem % blockdim;
	}
	
	public int getTag () {
		return tag;
	}
	
	public int getBlock () {
		return block;
	}
	
	public int getAddress () {
		return address;
	}
	
	public void setTag (int t) {
		tag = t;
	}
	
	public void setBlock (int b) {
		block = b;
	}
	
	public void setAddress (int a) {
		address = a;
	}
}

class SetAssociativeMappingElement extends MappingElement {
	private int tag;
	private int set;
	private int address;
	
	public SetAssociativeMappingElement (int t, int s, int a) {
		tag = t;
		set = s;
		address = a;
	}
	
	public SetAssociativeMappingElement (int elem, int blockdim, int setDim, int out) {
		tag = elem / blockdim / setDim;
		set = (elem / blockdim) % setDim;
		address = elem % blockdim;
	}
	
	public int getTag () {
		return tag;
	}
	
	public int getSet () {
		return set;
	}
	
	public int getAddress () {
		return address;
	}
	
	public void setTag (int t) {
		tag = t;
	}
	
	public void setSet (int s) {
		set = s;
	}
	
	public void setAddress (int a) {
		address = a;
	}
}

class CacheHandler {
	private int cacheType;
	private int cacheDimData;
	private int cacheDimInstruction;
	private int blockDimData;
	private int blockDimInstruction;
	private int cacheMapping;
	private int replaceMode;
	private File cacheFile;
	private OutputData out;
	
	private ReferenceData refData;
	private MemoryHandler myMemoryHandler;
	private CacheMemory DataCache;
	private CacheMemory InstructionCache;
	
	public CacheHandler () {
		DataCache = new CacheMemory ();
		InstructionCache = new CacheMemory ();
		DataCache.setBlockDim (0);
		InstructionCache.setBlockDim (0);
		DataCache.setCacheDim (0);
		InstructionCache.setCacheDim (0);
		DataCache.setCacheMapping (0);
		InstructionCache.setCacheMapping (0);
		DataCache.setReplaceMode (0);
		InstructionCache.setReplaceMode (0);
	}
	
	public void setCacheType (int ct) {
		cacheType = ct;
	}

	public void setCacheDimData (int cd) {
		cacheDimData = cd;
		DataCache.setCacheDim (cd);
	}
	
	public void setCacheDimInstruction (int ci) {
		cacheDimInstruction = ci;
		InstructionCache.setCacheDim (ci);
	}
	
	public void setBlockDimData (int bd) {
		blockDimData = bd;
		DataCache.setBlockDim (bd);
	}

	public void setBlockDimInstruction (int bi) {
		blockDimInstruction = bi;
		InstructionCache.setBlockDim (bi);
	}

	public void setCacheMapping (int cm) {
		cacheMapping = cm;
		DataCache.setCacheMapping (cm);
		InstructionCache.setCacheMapping (cm);
	}

	public void setReplaceMode (int rm) {
		replaceMode = rm;
		DataCache.setReplaceMode (rm);
		InstructionCache.setReplaceMode (rm);
	}

	public void setFile (File f) {
		cacheFile = f;
		myMemoryHandler = new MemoryHandler (cacheFile);
	}

	public int getCacheType () {
		return cacheType;
	}

	public int getCacheDim () {
		return cacheDimData;
	}

	public int getBlockDim () {
		return blockDimData;
	}

	public int getCacheMapping () {
		return cacheMapping;
	}
	
	public OutputData getOutData () {
		return out;
	}

	public void simulate () {
		DataCache.resetInfo ();
		InstructionCache.resetInfo ();
		DataCache.setLRUTable ();
		InstructionCache.setLRUTable ();
		myMemoryHandler.setSeek ();
		
		do {
			refData = myMemoryHandler.getNextElem ();
			if (refData == null)
				break;
			addReferenceData ();
		} while (refData != null);

		out = getOutputData ();
		System.out.println ("\t\t========== RESULTS ==========");
		System.out.println ("\n\t\t Total references : "+out.getRefNum());
		System.out.println ("\n\t\t Instruction references : "+out.getInstrNum()+"   Instruction Misses : "+out.getInstrMissNum());
		System.out.println ("\n\t\t Data references : "+out.getDataNum()+"   Data Misses: "+out.getDataMissNum());
		System.out.println ("\n\t\t Read data: "+out.getReadNum()+"   Read Misses : "+out.getReadMissNum());
		System.out.println ("\n\t\t Write Data : "+out.getWriteNum()+"    Write Misses : "+out.getWriteMissNum());
		System.out.println ("\n\n");
		DataCache.cleanMemory ();
		InstructionCache.cleanMemory ();
	}
	
	private void addReferenceData () {
		int dataType;
		int address;
		
		dataType = refData.getLabel ();
		address = refData.getAddress ();
		
		if (dataType == 2) { // instruction fetch
			if (cacheType == 0)
				DataCache.instrCache (address);
			else
				InstructionCache.instrCache (address);
		}
		
		if (dataType == 1)   // write
			DataCache.writeToCache (address);
		
		if (dataType == 0)   // read
			DataCache.readFromCache (address);
	}

	private OutputData getOutputData () {
		int instrNum, instrMiss, readNum, writeNum, readMiss, writeMiss;
		int refNum, dataNum, dataMiss;

		if (cacheType == 0) {
			instrNum = DataCache.getInstrDemands ();
			instrMiss = DataCache.getInstrMisses ();
		}
		else {
			instrNum = InstructionCache.getInstrDemands ();
			instrMiss = InstructionCache.getInstrMisses ();
		}

		readNum = DataCache.getReadDemands ();
		writeNum = DataCache.getWriteDemands ();
		readMiss = DataCache.getReadMisses ();
		writeMiss = DataCache.getWriteMisses ();
		dataNum = readNum + writeNum;
		refNum = instrNum + dataNum;
		dataMiss = readMiss + writeMiss;
		out = new OutputData (refNum, instrNum, dataNum, readNum, writeNum, instrMiss, dataMiss, readMiss, writeMiss);

		return out;
	}
}

class CacheMemory {
	private int cacheDim;
	private int blockDim;
	private int setDim;
	private int replaceMode;
	private int cacheMapping;
	private int cacheType;
	private int instrdemands;
	private int instrmisses;
	private int readdemands;
	private int writedemands;
	private int readmisses;
	private int writemisses;

	private int[] LRUTable;
	private int FIFOBlock;
	private MappingElement[] cacheMemory;

	private int cacheindex;

	public CacheMemory (int cd, int bd, int sd, int rm, int cm, int ct) {
		cacheDim = cd;
		blockDim = bd;
		setDim = sd;
		replaceMode = rm;
		cacheMapping = cm;
		cacheType = ct;
		LRUTable = new int [cacheDim / blockDim];
		cacheMemory = new MappingElement [cacheDim];
	}

	public CacheMemory (int ct) {
		cacheType = ct;
	}

	public CacheMemory () {
		instrdemands = 0;
		instrmisses = 0;
		readdemands = 0;
		writedemands = 0;
		readmisses = 0;
		writemisses = 0;
		FIFOBlock = 0;
		setDim = 8;
	}
	
	public void resetInfo () {
		instrdemands = 0;
		instrmisses = 0;
		readdemands = 0;
		writedemands = 0;
		readmisses = 0;
		writemisses = 0;
		FIFOBlock = 0;
	}

	public void setCacheType (int ct) {
		cacheType = ct;
	}

	public void setCacheDim (int cd) {
		cacheDim = 8 * 1024;

		for (int i = 0; i < cd; i++)
			cacheDim *= 2;

		cacheMemory = new MappingElement [cacheDim];
	}
	
	public void setBlockDim (int bd) {
		blockDim = 1;

		for (int i = 0; i < bd; i++)
			blockDim *= 2;
	}
	
	public void cleanMemory () {
		for (int i = 0; i < cacheDim; i++)
			cacheMemory[i] = null;
	}
	
	public void setLRUTable () {
		LRUTable = new int [cacheDim / blockDim];
	}

	public void setSetDim (int sd) {
		setDim = 2;

		for (int i = 0; i < sd; i++)
			setDim *= 2;
	}

	public void setCacheMapping (int cm) {
		cacheMapping = cm;
	}

	public void setReplaceMode (int rm) {
		if (rm == 0)
			setDim = 8;
		replaceMode = rm;
	}

	public int getCacheType () {
		return cacheType;
	}

	public int getCacheDim () {
		return cacheDim;
	}

	public int getBlockDim () {
		return blockDim;
	}

	public int getSetDim () {
		return setDim;
	}

	public int getCacheMapping () {
		return cacheMapping;
	}

	public int getReplaceMode () {
		return replaceMode;
	}

	public int getInstrDemands () {
		return instrdemands;
	}

	public int getReadDemands () {
		return readdemands;
	}
	
	public int getWriteDemands () {
		return writedemands;
	}
	
	public int getInstrMisses () {
		return instrmisses;
	}
	
	public int getReadMisses () {
		return readmisses;
	}
	
	public int getWriteMisses () {
		return writemisses;
	}
	
	public void instrCache (int element) {
		instrdemands++;
		instrmisses += operateCache (element);
	}
	
	public void writeToCache (int element) {
		writedemands++;
		writemisses += operateCache (element);
	}
	
	public void readFromCache (int element) {
		readdemands++;
		readmisses += operateCache (element);
	}
	
	private int operateCache (int element) {
		if (cacheMapping == 2) {
			DirectMappingElement de = new DirectMappingElement (element, cacheDim, blockDim, 0);
			return operateDirect (de);
		}
		if (cacheMapping == 1) {
			AssociativeMappingElement ae = new AssociativeMappingElement (element, blockDim, 0);
			return operateAssociative (ae);
		}
		if (cacheMapping == 0) {
			SetAssociativeMappingElement se = new SetAssociativeMappingElement (element, blockDim, setDim, 0);
			return operateSetAssociative (se);
		}
		return 1;
	}

	private int operateAssociative (AssociativeMappingElement cachein) {
		int cacheindex = cacheDim;
		int neededblock = -1;
		int miss = 1;

		for (int i = 0; i < cacheDim; i += blockDim) {
			if (cacheMemory[i] == null) {
				cacheindex = i;
				break;
			}

			if (cachein.getTag () == ((AssociativeMappingElement) cacheMemory[i]).getTag ()) {
				neededblock = i / blockDim;
				miss = 0;
				break;
			}
		}
		
		if (miss == 1) {
			if (cacheindex == cacheDim) {
				if (replaceMode == 0)
					assocRandomReplace (cachein);
				if (replaceMode == 1) {
					assocLRUReplace (cachein);
					assocLRUTable (cachein.getTag ());
				}
				if (replaceMode == 2)
					assocFIFOReplace (cachein);
			}
			else
				assocCopyToCache (cacheindex / blockDim, cachein);
		}
		else
			assocLRUTable (neededblock);
		
		return miss;
	}
	
	private void assocRandomReplace (AssociativeMappingElement cachein) {
		int randomblock = (int) (Math.random () * (cacheDim / blockDim - 1));
		
		assocCopyToCache (randomblock, cachein);
	}
	
	private void assocLRUTable (int neededblock) {
		for (int i = 0; i < cacheDim / blockDim; i++) {
			if (LRUTable[i] > LRUTable[neededblock]) {
				LRUTable[i]++;
			}
			if (i == neededblock)
				LRUTable[i] = 0;
		}
	}
	
	private void assocLRUReplace (AssociativeMappingElement cachein) {
		int replaceblock = 0;
		
		for (int i = 0; i < cacheDim / blockDim; i++) {
			if (LRUTable[i] > LRUTable[replaceblock])
				replaceblock = i;
		}
		assocCopyToCache (replaceblock, cachein);
	}
	
	private void assocFIFOReplace (AssociativeMappingElement cachein) {
		assocCopyToCache (FIFOBlock, cachein);
		FIFOBlock = (FIFOBlock + 1) % (cacheDim / blockDim);
	}
	
	private void assocCopyToCache (int replaceblock, AssociativeMappingElement cachein) {
		int replaceblockstart = replaceblock * blockDim;
		int meminblock = cachein.getTag ();
		
		for (int i = 0; i < blockDim; i++) {
			cacheMemory[replaceblockstart + i] = new AssociativeMappingElement (meminblock, i);
		}
	}
	
	private int operateDirect (DirectMappingElement cachein) {
		cacheindex = cacheDim;
		int miss = 0;

		if (cacheMemory[cachein.getBlock () * blockDim] == null) {
			cacheindex = cachein.getBlock () * blockDim;
			miss = 1;
		}
		
		else {
			if (cachein.getTag () != ((DirectMappingElement) cacheMemory[cachein.getBlock() * blockDim]).getTag ())
				miss = 1;
		}

		if (miss == 1) {
			directCopyToCache (cachein);
		}

		return miss;
	}

	private void directCopyToCache (DirectMappingElement cachein) {
		int replaceblockstart = cachein.getBlock () * blockDim;
		
		for (int i = 0; i < blockDim; i++) {
			cacheMemory[replaceblockstart + i] = new DirectMappingElement (cachein.getTag (), cachein.getBlock (), i);
		}
	}

	private int operateSetAssociative (SetAssociativeMappingElement cachein) {
		int replaceOffset = 2;
		int neededblock = -1;
		int miss = 1;
		boolean isnull = false;
		int addressStartSet = cacheDim / setDim * cachein.getSet ();
		int blocksperSet = cacheDim / setDim / blockDim;

		for (int i = addressStartSet; i < addressStartSet + blocksperSet * blockDim; i += blockDim) {
			if (cacheMemory[i] == null) {
				miss = 1;
				replaceOffset = (i - addressStartSet) / blockDim;
				isnull = true;
				break;
			}

			if (cachein.getTag () == ((SetAssociativeMappingElement) cacheMemory[i]).getTag ()) {
				miss = 0;
				break;
			}
		}

		if (miss == 1) {
			if (isnull == false) {
				/*if (replaceMode == 0)*/
					setAssocRandomReplace (cachein);
				/*if (replaceMode == 1) {
					assocLRUTable (neededblock);
					assocLRUReplace (cachein);
				}
				if (replaceMode == 2)
					assocFIFOReplace (cachein);*/
			}
			else
				setAssocCopyToCache (cachein, replaceOffset);
		}

		return miss;
	}
	
	void setAssocRandomReplace (SetAssociativeMappingElement cachein) {
		int setStartAddress = cacheDim / setDim * cachein.getSet ();
		int blocksperSet = cacheDim / setDim / blockDim;
		
		int offset = (int) (Math.random () * blocksperSet);
		
		setAssocCopyToCache (cachein, offset);
		
	}
	
	void setAssocCopyToCache (SetAssociativeMappingElement cachein, int offset) {
		int setStartAddress = cacheDim / setDim * cachein.getSet ();
		
		for (int i = setStartAddress + offset * blockDim; i < setStartAddress + (offset + 1) * blockDim; i++)
			cacheMemory[i] = new SetAssociativeMappingElement (cachein.getTag (), cachein.getSet (), i);
	}
}